<?php
//OK to PUBLISH
error_reporting(E_ALL);
ini_set('display_errors', 0);
date_default_timezone_set('UTC');

// relative directory paths against directory where publication.php is located
$target_dir_path = './';
$weblog_dir_path = './##dir_prefix##logs/';

//======================
set_exception_handler('pub_exception_handler');
set_error_handler('pub_error_handler');

$full_dir_paths = null;
$revdata = null;
$dir_prefix = null;

run();
//======================

function run()
{
    $VALID_ACTIONS = array('get_seed', 'send_data', 'get_test_data');

    $action = isset($_REQUEST['action']) && is_string($_REQUEST['action']) ? strtolower($_REQUEST['action']) : '';

    if (!$action)
    {
        write_log_data_and_die(array('success' => 0, 'error' => 'Action is not defined'));
    }

    if (!in_array($action, $VALID_ACTIONS))
    {
        write_log_data_and_die(array('success' => 0, 'error' => 'Action is not valid'));
    }

    if ($action == 'get_seed')
    {
        $seed = run_get_seed();
        echo '##SEED##'.base64_encode($seed);
    }
    else
    if ($action == 'send_data')
    {
        $res = run_send_data();

        if ($res)
        {
            echo "OK\n";
        }
        else
        {
            echo "ER\n";
            echo "Target server failed to accept files.\n";
        }
    }
    else
    if ($action == 'get_test_data')
    {
        $res = run_get_test_data($test_data);

        if ($res)
        {
            echo "OK\n";
            echo json_encode($test_data, JSON_FORCE_OBJECT|JSON_PRETTY_PRINT);
        }
        else
        {
            echo "ER\n";
            echo "Target server failed to provide test data.\n";
        }
    }

    return;
}

//
// authentication
//
function run_get_seed()
{
    $seed = generate_seed_string(64);
    remove_old_seed_data();
    save_new_seed_data($seed);
    return $seed;
}

function generate_seed_string($length)
{
    $res = '';

    for ($i = 0; $i < $length; $i++)
    {
        // please do not use here 0 bytes, because results unpredictable
        $random_int = rand(1, 255);
        $ch = chr($random_int);
        if ($ch == '|') {
            $ch = '!';
        }
        $res .= $ch;
    }

    return $res;
}

function remove_old_seed_data()
{
    $MAX_AGE = 300;//in seconds

    // find all files having name like 'ZZ_SEED_xxx_yyy.txt';
    // and delete files older than $MAX_AGE
    $src_dir = dirname(__FILE__) . '/';
    $dir_handler = dir($src_dir);

    while (false !== ($name = $dir_handler->read()))
    {
        if ($name == '.' || $name == '..') {
            continue;
        }

        $file_path = $src_dir . $name;

        if (is_file($file_path))
        {
            if (preg_match("`ZZ_SEED_([0-9]+)_([0-9]+)\.txt`siU", $name, $m))
            {
                $file_mtime = intval($m[2]);

                if ((filemtime($file_path) + $MAX_AGE) < time())
                {
                    unlink($file_path);
                }
            }
        }
    }//while

    return;
}

function save_new_seed_data($seed)
{
    // build file name
    // format: ZZ_SEED_xxx_yyy.txt
    // where:
    // xxx - 10 numbers based on first 10 chars of seed
    // yyy - time()
    $file_name = 'ZZ_SEED_';

    $chars = str_split($seed);

    for ($i = 0; $i < 10; $i++)
    {
        $n = ord($chars[$i]) % 9;
        $file_name .= $n;
    }

    $file_name .= '_'.time().'.txt';

    $file_path = dirname(__FILE__) . '/' . $file_name;
    file_put_contents($file_path, $seed);

    return;
}

function decrypt_string($str_encrypted)
{
    $KEY_TYPE = 'public';

    $KEY_PUBLIC_PEM =
'
-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA1kGqvM0XOq14oBdLwkDz
ewXZP58N6llDVGhrdeGhNQbL6MFkTLWHtC7a+tGGQDo2mjBXZdAyOl6EC3d11q02
CpyTgXfSFexjweMHkhFAJxk1zvzmSaJadGjbQ2dGC1NzyuQuAY8n+jvFXA/7orqK
oeQUVbVa/DAyH9YnbnBRDS81gVSUh2xRGhU6s8/VCXvfP+36nIL7hhkRuxFW6qFD
g+o/exU5kJCzBMJQHpwB84CCr0BpBeo1MoSFRG1TXutQf1BXfHCpXIVLDzwr3vsb
dn4kSDgnckh0Lmdjb2RdEHSFauxWgs6TaWRFovY4ClUnCppbyHk4afwivZjlfM0e
zwIDAQAB
-----END PUBLIC KEY-----
';

    //Block size for encryption block cipher
    $ENCRYPT_BLOCK_SIZE = 200;// this for 2048 bit key for example, leaving some room
    //Block size for decryption block cipher
    $DECRYPT_BLOCK_SIZE = 256;// this again for 2048 bit key

    $key_res = openssl_pkey_get_public($KEY_PUBLIC_PEM);

    $str_encrypted = base64_decode(str_replace('#', '', $str_encrypted));
    $str_encrypted_parts = str_split($str_encrypted, $DECRYPT_BLOCK_SIZE);

    $str_plain = '';

    foreach ($str_encrypted_parts as $part_encrypted)
    {
        $part_plain = '';
        $dec_is_ok = openssl_public_decrypt($part_encrypted, $part_plain, $key_res, OPENSSL_PKCS1_PADDING);

        if ($dec_is_ok === false) {
            write_log_data_and_die(array('success' => 0, 'error' => 'Decryption failed'));
        }

        $str_plain .= $part_plain;
    }

    return $str_plain;
}

function strtohex($x)
{
    $s = '';

    foreach (str_split($x) as $c)
    {
        $s .= sprintf("%02X", ord($c));
    }

    return($s);
}

//
// publication
//
function run_send_data()
{
    global $full_dir_paths;
    global $dir_prefix;
    global $use_blv_sc;

    $str_data_encrypted = isset($_REQUEST['data']) && is_string($_REQUEST['data']) ? $_REQUEST['data'] : '';
    $domain_name = isset($_REQUEST['domain_name']) && is_string($_REQUEST['domain_name']) ? $_REQUEST['domain_name'] : '';
    $dir_prefix = isset($_REQUEST['dir_prefix']) && is_string($_REQUEST['dir_prefix']) ? $_REQUEST['dir_prefix'] : null;
    $use_blv_sc = isset($_REQUEST['use_blv_sc']) && intval($_REQUEST['use_blv_sc']) ? 1 : 0;

    // check if domain name is defined
    if (!strlen($domain_name)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Domain name not defined'));
    }

    // check if data is defined
    if (!strlen($str_data_encrypted)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Data not defined'));
    }

    // check if dir prefix is defined
    if (is_null($dir_prefix)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Dir prefix not defined'));
    }

    if (!validate_received_data($str_data_encrypted, $domain_name)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Data is incorrect'));
    }

    if (!get_content_file()) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Failed to get content file'));
    }

    return true;
}

function validate_received_data($str_data_encrypted, $domain_name)
{
    global $revdata;

    $str_data_plain = decrypt_string($str_data_encrypted);

    if (!strlen($str_data_plain)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Data is empty'));
    }

    $data = explode('|', $str_data_plain);

    if (!is_array($data) || count($data) != 6) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Data is incomplete'));
    }

    // extract all data from array
    $seed_plain = base64_decode($data[0]);
    $reverse_url = base64_decode($data[1]);
    $target_domain = base64_decode($data[2]);
    $file_key = $data[3];
    $str_signature = $data[4];
    $concat_hash = $data[5];

    //
    // validate seed
    //
    if (strlen($seed_plain) != 64) {
        return false;
    }

    // build file name (only part of it)
    // format: ZZ_SEED_xxx_yyy.txt
    // where:
    // xxx - 10 numbers based on first 10 chars of seed
    // yyy - time()
    $file_name = 'ZZ_SEED_';

    $chars = str_split($seed_plain);

    for ($i = 0; $i < 10; $i++)
    {
        $n = ord($chars[$i]) % 9;
        $file_name .= $n;
    }
    // build file name (only part of it) [end]

    // try to find file and get it's content
    $file_content = '';
    $src_dir = dirname(__FILE__) . '/';
    $dir_handler = dir($src_dir);

    while (false !== ($name = $dir_handler->read()))
    {
        if ($name == '.' || $name == '..') {
            continue;
        }

        $file_path = $src_dir . $name;

        if (is_file($file_path))
        {
            if (preg_match("`".$file_name."_([0-9]+)\.txt`siU", $name, $m))
            {
                $file_content = file_get_contents($file_path);
                $seed_file_name = $name;
                $seed_file_base_path = $src_dir;
                break;
            }
        }
    }//while

    $seed_is_valid = false;

    if (strlen($file_content) && $file_content === $seed_plain) {
        $seed_is_valid = true;
    }

    if (!$seed_is_valid) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Seed is invalid'));
    }
    // validate seed [end]

    //
    // validate other data
    //
    // domain name vs target domain
    if ($domain_name != $target_domain) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Domain name is inconsistent'));
    }

    // reverse url
    if (!strlen($reverse_url)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Reverse URL is empty'));
    }

    // file key
    if (!strlen($file_key)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'File key is empty'));
    }

    // signature
    if (!strlen($str_signature)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Signature is empty'));
    }

    // concat hash
    if (!strlen($concat_hash)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Concat hash is empty'));
    }
    // validate other data [end]

    // save uploaded file details to memory
    $revdata = array(
        'target_domain' => $target_domain,
        'reverse_url' => $reverse_url,
        'file_key' => $file_key,
        'str_signature' => $str_signature,
        'concat_hash' => $concat_hash,
    );

    return true;
}

function get_content_file()
{
    global $revdata, $full_dir_paths;

    //
    // get file content
    //
    // check if cURL is possible
    $is_curl_enabled = function_exists('curl_version') ? true : false;
    $is_allow_url_fopen_enabled = intval(ini_get('allow_url_fopen')) ? true : false;

    if (!$is_curl_enabled && !$is_allow_url_fopen_enabled) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Can not get file because cURL and remote fopen are disabled'));
    }

    $data = array(
        'd' => $revdata['target_domain'],
        'k' => $revdata['file_key'],
    );

    $file_content = '';

    if ($is_curl_enabled)
    {
        $file_content = get_content_via_curl($revdata['reverse_url'], $data);
    }
    else
    if ($is_allow_url_fopen_enabled)
    {
        $file_content = get_content_via_fopen($revdata['reverse_url'], $data);
    }

    if (strlen($file_content) < 100) {
        write_log_data_and_die(array('success' => 0, 'error' => 'File content length minimum is not met'));
    }
    // get file content [end]

    // create necessary directories
    $full_dir_paths = make_directories($revdata['target_domain']);
    $site_root = $full_dir_paths['site_root'];

    //
    // save file content
    //
    $new_file_path = $site_root . 'site.tar';

    if (is_file($new_file_path)) {
        unlink($new_file_path);
    }

    file_put_contents($new_file_path, $file_content);

    if (!is_file($new_file_path)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Failed to save file on disk'));
    }
     // save file content [end]

    //
    // validate hashes
    //
    // NB: 2nd param must be false; in that case we will get 32 char length string
    $file_hash = md5_file($new_file_path, false);
    // build hash on concatenation of signature and file hash
    $concat_hash = md5($revdata['str_signature'].'|'.$file_hash);

    if ($concat_hash != $revdata['concat_hash']) {
        unlink($new_file_path);
        write_log_data_and_die(array('success' => 0, 'error' => 'File hash is invalid'));
    }
    // validate hashes [end]

    //
    // do backup of index.php
    //
    $index_file_path = $site_root . 'index.php';

    if (is_file($index_file_path))
    {
        $index_file_content = file_get_contents($index_file_path);
        $special_string = '#BLV-PHP';

        if (stripos($index_file_content, $special_string) === false)
        {
            $backup_path = $site_root . 'index_blv_backup.php';

            if (!rename($index_file_path, $backup_path))
            {
                write_log_data_and_die(array('success' => 0, 'error' => 'Failed to backup index.php file'));
            }
        }
    }
    // do backup of index.php[end]

    preserve_existing_db_config($site_root);

    // extract .tar file content
    extract_files($site_root, $new_file_path);

    restore_existing_db_config($site_root);

    //
    // update .htaccess
    //
    $hta_file_path = $site_root . '.htaccess';

    // if file not existing then do touch()
    // this way we will make sure that file always exists; that's needed for further checks
    if (!is_file($hta_file_path)) {
        // do we need here $is_created handling ?
        $is_created = @touch($hta_file_path);
    }

    if (!is_file($hta_file_path))
    {
        if (!is_writable($site_root)) {
            write_log_data_and_die(array('success' => 0, 'error' => 'Home dir is not writable'));
        } else {
            write_log_data_and_die(array('success' => 0, 'error' => 'Home dir`s HTA file does not exist and is not writable'));
        }
    }

    $has_hta_rules = false;

    // if HTA is not writable then try to fix it through recreating
    if (!is_writable($hta_file_path))
    {
        $tmp_file_path = $site_root . '.htaccess_blv_tmp';

        if (copy($hta_file_path, $tmp_file_path))
        {
            unlink($hta_file_path);
            rename($tmp_file_path, $hta_file_path);
        }
    }

    // check if HTA is writable or not 2nd time
    if (!is_writable($hta_file_path))
    {
        $has_hta_rules = check_has_hta_rules($hta_file_path);

        if (!$has_hta_rules) {
            write_log_data_and_die(array('success' => 0, 'error' => 'Home dir`s HTA file exists, but is not writable'));
        }
    }

    if (!$has_hta_rules)
    {
        update_hta_file($hta_file_path);

        $has_hta_rules = check_has_hta_rules($hta_file_path);

        if (!$has_hta_rules) {
            write_log_data_and_die(array('success' => 0, 'error' => 'Home dir`s HTA file is writable, but updated file is missing some(or all) of our rules'));
        }
    }
    // update .htaccess [end]

    //
    // add .htaccess to logs directory
    //
    $hta_file_path = $full_dir_paths['publication_logs'] . '.htaccess';

    $hta_rules = "
#BLV-START-1 DO NOT MANUALY CHANGE ANYTHING UNTIL BLV-END
<filesmatch \".*\">
    Order Allow,Deny
    Deny from all
</filesmatch>

Options All -Indexes
#BLV-END-1
";
    if (!file_put_contents($hta_file_path, $hta_rules)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Failed to write log dir`s HTA file'));
    }
    // add .htaccess to logs directory

    //
    // update blv_initial.php
    //
    if (!update_blv_initial()) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Failed to update blv_initial.php'));
    }
    // update blv_initial.php [end]

    return true;
}

function check_has_hta_rules($hta_file_path)
{
    $res = true;

    $current_hta_rules = file_get_contents($hta_file_path);

    // check "generic" sections presence
    $our_hta_rules = build_our_hta_rules();

    foreach ($our_hta_rules as $k => $section)
    {
        if (strpos($current_hta_rules, $section) === false)
        {
            $res = false;
            break;
        }
    }

    // check "DirectoryIndex" section presence
    $regex = "`^[ \t]*DirectoryIndex[ \t]+index\.php`";

    if (!preg_match($regex, $current_hta_rules, $m))
    {
        $res = false;
    }

    return $res;
}

function update_hta_file($hta_file_path)
{
    // get current content of .htaccess; do not modify it
    $current_hta_rules = file_get_contents($hta_file_path);
    // get current content for modification
    $new_hta_rules = $current_hta_rules;

    $our_hta_rules = build_our_hta_rules();

    //
    // add/update "generic" sections
    //
    if (!strlen($new_hta_rules))
    {
        $new_hta_rules = implode("\n", $our_hta_rules);
    }
    else
    {
        foreach ($our_hta_rules as $k => $section)
        {
            $str_start = "#BLV-START-{$k} DO NOT MANUALY CHANGE ANYTHING UNTIL BLV-END\n";
            $str_end = "#BLV-END-{$k}\n";

            $regex = "`{$str_start}.+{$str_end}`s";

            $m = array();

            if (!preg_match($regex, $new_hta_rules, $m))
            {
                $new_hta_rules = $section."\n".$new_hta_rules;
            }
            else
            {
                if ($m[0] != $section)
                {
                    $s2 = str_replace('=$', '=\$', $section);
                    $new_hta_rules = preg_replace($regex, $s2, $new_hta_rules);
                }
                else
                {
                    // section IS found but content is same! no replacement to do!
                }
            }
        }//foreach
    }
    // add/update "generic" sections [end]

    //
    // add/update "DirectoryIndex" section
    //
    $regex = "`^[ \t]*DirectoryIndex[ \t]+index\.php`";

    if (!preg_match($regex, $new_hta_rules, $m))
    {
        $s1 = "DirectoryIndex index.php\n\n";
        $new_hta_rules = $s1.$new_hta_rules;
    }
    // add/update "DirectoryIndex" section [end]

    // update file if needed
    if ($new_hta_rules != $current_hta_rules)
    {
        if (backup_existing_hta_file())
        {
            $res = file_put_contents($hta_file_path, $new_hta_rules);

            if ($res === false) {
                write_log_data_and_die(array('success' => 0, 'error' => 'Failed to write into HTA file'));
            }
        }
        else
        {
            write_log_data_and_die(array('success' => 0, 'error' => 'Failed to copy HTA file for backup'));
        }
    }

    return;
}

function backup_existing_hta_file()
{
    global $full_dir_paths;

    $base_path = $full_dir_paths['site_root'];

    $old_file_path = $base_path . '.htaccess';
    $new_file_path = $base_path . '.htaccess_blv_backup';

    // check if old file content is non-empty; if file size is 0 then we do not create backup
    if (filesize($old_file_path) === 0) {
        return true;
    }

    $res = copy($old_file_path, $new_file_path);

    return $res;
}

function build_our_hta_rules()
{
    $sections = array();

    $pages_list = get_pages_list();

    if ($pages_list)
    {
        $str_rule1 = "^(".implode('|', $pages_list).")$";

        $s1  = "#BLV-START-1 DO NOT MANUALY CHANGE ANYTHING UNTIL BLV-END\n";
        $s1 .= "<IfModule mod_rewrite.c>\n";
        $s1 .= "  RewriteEngine on\n";
        $s1 .= "  RewriteCond %{REQUEST_FILENAME} !-f\n";
        $s1 .= "  RewriteCond %{REQUEST_FILENAME} !-d\n";
        $s1 .= "  RewriteRule $str_rule1 index.php?pg=$1 [L,QSA]\n";
        $s1 .= "</IfModule>\n";
        $s1 .= "#BLV-END-1\n";

        $sections[1] = $s1;
    }

    return $sections;
}

function update_blv_initial()
{
    global $full_dir_paths, $use_blv_sc;

    $file_path = $full_dir_paths['site_root'] . 'blv_initial.php';
    $file_path_tmp = $full_dir_paths['site_root'] . '_tmp_blv_initial.php';

    if ($use_blv_sc)
    {
        if (!is_file($file_path_tmp)) {
            write_log_data_and_die(array('success' => 0, 'error' => 'blv_initial.php temp version is not found'));
        }

        if (!rename($file_path_tmp, $file_path)) {
            write_log_data_and_die(array('success' => 0, 'error' => 'Failed to publish blv_initial.php'));
        }
    }
    else
    {
        if (is_file($file_path))
        {
            if (!unlink($file_path)) {
                write_log_data_and_die(array('success' => 0, 'error' => 'Failed to remove blv_initial.php'));
            }
        }
    }

    return true;
}

function get_pages_list()
{
    global $full_dir_paths;
    global $dir_prefix;

    $res = array();

    // find all files having name like 'ZZ_SEED_xxx_yyy.txt';
    // and delete files older than $MAX_AGE
    $src_dir = $full_dir_paths['site_root'] . $dir_prefix . 'pages/';
    $dir_handler = dir($src_dir);

    while (false !== ($name = $dir_handler->read()))
    {
        if ($name == '.' || $name == '..') {
            continue;
        }

        $file_path = $src_dir . $name;

        if (is_file($file_path))
        {
            if (preg_match("`([^.]+)\.php`siU", $name, $m))
            {
                $res[] = $m[1];
            }
        }
    }//while

    return $res;
}

function get_content_via_curl($url, $data)
{
    $curl_headers = array(
        'User-Agent: Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.2.12) Gecko/20101026 Firefox/3.6.12',
        'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
        'Accept-Language: en-us,en;q=0.5',
        'Accept-Charset: ISO-8859-1,utf-8;q=0.7,*;q=0.7',
        'Accept-Encoding: deflate',
        'Keep-Alive: 115',
        'Connection: keep-alive',
    );

    $url_with_data = $url.'?'.http_build_query($data);

    $ch = curl_init($url_with_data);

    curl_setopt($ch, CURLOPT_HTTPHEADER, $curl_headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);
    curl_setopt($ch, CURLOPT_POST, false);
    // The number of seconds to wait while trying to connect. Use 0 to wait indefinitely.
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30);
    // The maximum number of seconds to allow cURL functions to execute.
    curl_setopt($ch, CURLOPT_TIMEOUT, 120);

    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);

    $curl_errors = curl_error($ch);

    if ($response === false)
    {
        $msg = "cURL request failed. Response is empty. cURL error:\n".$curl_errors."\nResponse:\n{$response}\nHTTP code:\n".$http_code."\n";
        write_log_data_and_die(array('success' => 0, 'error' => $msg));
    }
    else
    if ($http_code != '200')
    {
        $msg = "cURL request failed. HTTP code is not 200. cURL error:\n".$curl_errors."\nResponse:\n{$response}\nHTTP code:\n".$http_code."\n";
        write_log_data_and_die(array('success' => 0, 'error' => $msg));
    }

    if (strlen($response) < 100)
    {
        $msg = "cURL request failed. Content length minimum is not met. cURL error:\n".$curl_errors."\nResponse:\n{$response}\nHTTP code:\n".$http_code."\n";
        write_log_data_and_die(array('success' => 0, 'error' => $msg));
    }

    // response is file content which we expect from reverse url
    return $response;
}

function get_content_via_fopen($url, $data)
{
    $curl_headers = array(
        'User-Agent: Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.2.12) Gecko/20101026 Firefox/3.6.12',
        'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
        'Accept-Language: en-us,en;q=0.5',
        'Accept-Charset: ISO-8859-1,utf-8;q=0.7,*;q=0.7',
        'Accept-Encoding: deflate',
        'Keep-Alive: 115',
        'Connection: keep-alive',
    );

    $url_with_data = $url.'?'.http_build_query($data);

    // create a stream
    $opts = array(
        'http' => array(
            'method' => 'GET',
            'header' => implode("\r\n", $curl_headers),
            'timeout' => 120,
        ),
    );

    $context = stream_context_create($opts);

    try
    {
        // open the file using the HTTP headers set above
        $file_content = file_get_contents($url_with_data, false, $context);
    }
    catch (Exception $e)
    {
        $msg = "fopen request failed. Exception occured. Exception message:\n".$e->getMessage()."\n";
        write_log_data_and_die(array('success' => 0, 'error' => $msg));
    }

    if ($file_content === false)
    {
        write_log_data_and_die(array('success' => 0, 'error' => 'fopen request failed. Content is empty'));
    }

    return $file_content;
}

function make_directories($domain_name)
{
    global $target_dir_path;
    global $weblog_dir_path;
    global $dir_prefix;

    $current_dir_path = dirname(__FILE__) . '/';

    $full_dir_paths = array(
        'site_root' => clean_dir_path($current_dir_path . $target_dir_path),
        'publication_logs' => clean_dir_path($current_dir_path . $weblog_dir_path),
    );

    foreach ($full_dir_paths as $path_key => $full_dir_path)
    {
        $full_dir_path = str_replace('##domain_name##', $domain_name, $full_dir_path);

        if ($path_key == 'publication_logs') {
            $full_dir_path = str_replace('##dir_prefix##', $dir_prefix, $full_dir_path);
        }

        $full_dir_paths[$path_key] = $full_dir_path;

        // check if directory already exists
        if (is_dir($full_dir_path)) {
            continue;
        }

        // create directory. 3rd param = TRUE and that makes directory creation recursive
        $res = mkdir($full_dir_path, 0755, true);

        if (!$res)
        {
            write_log_data_and_die(array('success' => 0, 'error' => "Can not create directory {$full_dir_path}"));
        }
    }//foreach

    return $full_dir_paths;
}

function clean_dir_path($dir_path)
{
    $res = str_replace('/./', '/', $dir_path);
    $res = str_replace('//', '/', $res);

    return $res;
}

function preserve_existing_db_config($base_dir)
{
    global $dir_prefix;

    $dir_path = $base_dir . $dir_prefix . 'includes/';
    $src_file_path = $dir_path . 'db_config.php';
    $dst_file_path = $dir_path . 'db_config_preserved.php';

    if (file_exists($src_file_path) && is_file($src_file_path))
    {
        copy($src_file_path, $dst_file_path);
    }

    return;
}

function restore_existing_db_config($base_dir)
{
    global $dir_prefix;

    $dir_path = $base_dir . $dir_prefix . 'includes/';
    $src_file_path = $dir_path . 'db_config_preserved.php';
    $dst_file_path = $dir_path . 'db_config.php';

    if (file_exists($src_file_path) && is_file($src_file_path))
    {
        rename($src_file_path, $dst_file_path);
    }

    return;
}

function write_log_data_and_die($data)
{
    if (!isset($data['_REQUEST'])) {
        $data['_REQUEST'] = $_REQUEST;
    }
    write_log_data($data);
    echo "ER\n";
    echo $data['error']."\n";

//    $d = $_REQUEST;
//    if (isset($d['site'])) {
//        $d['site'] = '...file content truncated...';
//    }
//    echo "\n[REQUEST DATA BEGIN]\n".print_r($d, true)."[REQUEST DATA END]\n";
    die();
}

//
// write to file log data
//
function write_log_data($data = array())
{
    global $full_dir_paths;

    // make data dump
    $data = prepare_log_data($data);
    $str_data = print_r($data, true);

    $text  = date('d/m/Y H:i:s');
    $text .= '|'.$_SERVER['REMOTE_ADDR'].'|'.($data['success'] ? 'OK' : 'ERROR');
    $text .= "\n".$str_data;
    $text .= "========================================\n";

    $dir_path = is_array($full_dir_paths) && isset($full_dir_paths['publication_logs']) ? $full_dir_paths['publication_logs'] : '';

    if (!$dir_path || !is_dir($dir_path)) {
        return;
    }

    $file_name = 'publication_log.txt';
    $file_path = $dir_path . $file_name;

    rotate_log_file($dir_path, $file_name, 1000000);

    $fh = fopen($file_path, 'a');
    fwrite($fh, $text);
    fclose($fh);

    return;
}

function prepare_log_data($data)
{
    // list of valid variable names which we want to leave in data
    $remove_list = array('password', 'data', 'file_content');

    if (!isset($data['_REQUEST'])) {
        return $data;
    }

    foreach ($data['_REQUEST'] as $k => $v) {
        if (in_array($k, $remove_list)) {
            unset($data['_REQUEST'][$k]);
        }
    }

    return $data;
}

function rotate_log_file($dir_path, $file_name, $max_size = 1000000)
{
    $file_path = $dir_path . $file_name;

    if (file_exists($file_path))
    {
        if (filesize($file_path) > $max_size)
        {
            $s1 = '_' . date('y_m_d_H_i') . '.txt';
            $new_file_name = str_replace('.txt', $s1, $file_name);
            $new_file_path = $dir_path . $new_file_name;
            rename($file_path, $new_file_path);

            @chmod($new_file_path, 0775);
        }
    }

    return;
}

function extract_files($target_dir_path, $src_file_path)
{
    try
    {
        $phar = new PharData($src_file_path);
        // extract all files, and overwrite
        $phar->extractTo($target_dir_path, null, true);
    }
    catch (Exception $e)
    {
        echo 'failed to extract files. exception occured: '.$e->getMessage();
    }

    // update permissions of all files
    $GLOBALS['archive_base_path'] = 'phar://' . realpath($src_file_path) . '/';
    update_extracted_files_permissions($target_dir_path, $src_file_path);
}

function update_extracted_files_permissions($target_dir_path, $src_file_path)
{
    $archive = new PharData($src_file_path);

    foreach ($archive as $file)
    {
        if ($file->isFile())
        {
            $archive_full_path = $file->getPathname();
            $archive_relative_path = str_replace($GLOBALS['archive_base_path'], '', $archive_full_path);
            $extracted_full_path = $target_dir_path . $archive_relative_path;

            if (is_file($extracted_full_path))
            {
                //@TODO: add special treatment of .htaccess files
                @chmod($extracted_full_path, 0644);
            }
        }
        else
        if ($file->isDir())
        {
            update_extracted_files_permissions($target_dir_path, $file->getPathname());
        }
    }

    return;
}

function pub_exception_handler(Exception $e)
{
    // disable error capturing to avoid recursive errors
    restore_error_handler();
    restore_exception_handler();

    // use the most primitive way to log error
    $msg = get_class($e).': '.$e->getMessage().' ('.$e->getFile().':'.$e->getLine().")\n";
    $msg .= $e->getTraceAsString()."\n";
    $msg .= '$_SERVER='.var_export($_SERVER,true);

    write_log_data_and_die(array('success' => 0, 'error' => "PHP EXCEPTION.\nDetails below:\n".$msg));
}

function pub_error_handler($errno, $errstr, $errfile, $errline)
{
    // disable error capturing to avoid recursive errors
    restore_error_handler();
    restore_exception_handler();

    $log = "$errstr ($errfile:$errline)\nStack trace:\n";

    $trace = debug_backtrace();

    foreach ($trace as $i=>$t)
    {
        if (!isset($t['file']))
            $t['file'] = 'unknown';

        if (!isset($t['line']))
            $t['line'] = 0;

        if (!isset($t['function']))
            $t['function'] = 'unknown';

        $log .= "#$i {$t['file']}({$t['line']}): ";

        if (isset($t['object']) && is_object($t['object']))
            $log .= get_class($t['object']).'->';

        $log .= "{$t['function']}()\n";
    }

    write_log_data_and_die(array('success' => 0, 'error' => "PHP ERROR.\nDetails below:\n".$log));
}

function run_get_test_data(&$data)
{
    global $target_dir_path;
    global $dir_prefix;

    //
    // validate request data
    //
    $domain_name = isset($_REQUEST['domain_name']) && is_string($_REQUEST['domain_name']) ? $_REQUEST['domain_name'] : '';
    $dir_prefix = isset($_REQUEST['dir_prefix']) && is_string($_REQUEST['dir_prefix']) ? $_REQUEST['dir_prefix'] : null;

    // check if domain name is defined
    if (!strlen($domain_name)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Domain name not defined'));
    }

    // check if dir prefix is defined
    if (is_null($dir_prefix)) {
        write_log_data_and_die(array('success' => 0, 'error' => 'Dir prefix not defined'));
    }

    //
    // define proper path to target directory
    //
    $current_dir_path = dirname(__FILE__) . '/';
    $site_root = clean_dir_path($current_dir_path . $target_dir_path);
    $site_root = str_replace('##domain_name##', $domain_name, $site_root);

    //
    // collect data
    //
    test_get_env_details($data, $site_root);
    test_get_dir_listing($data, $site_root);

    return true;
}

function test_get_env_details(&$data, $site_root)
{
    global $dir_prefix;

    // phpversion
    $data['phpversion'] = phpversion();

    // phpinfo
    ob_start();
    phpinfo();
    $data['phpinfo'] = ob_get_clean();

    // curl, fopen
    $data['is_curl_enabled'] = function_exists('curl_version') ? 1 : 0;
    $data['is_allow_url_fopen_enabled'] = intval(ini_get('allow_url_fopen')) ? 1 : 0;

    // PharData
    $data['is_phardata_enabled'] = class_exists('PharData') ? 1 : 0;

    // disk space
    $data['hdd_total_size'] = disk_total_space($site_root);
    $data['hdd_free_size'] = disk_free_space($site_root);

    // website home (root) dir read/write permissions
    $data['home_readable'] = is_readable($site_root) ? 1 : 0;
    $data['home_writable'] = is_writable($site_root) ? 1 : 0;

    // check main files
    $keys1 = array(
        'rbt' => $site_root . 'robots.txt',
        'hta' => $site_root . '.htaccess',
        'idx' => $site_root . 'index.php',
        'blv' => $site_root . 'blv_initial.php',
    );

    $data['files'] = array();

    foreach ($keys1 as $k => $file_path)
    {
        $special_str_key = '';

        if ($k == 'hta') {
            $special_str_key = 'str1';
        }
        else
        if ($k == 'idx' || $k == 'blv')
        {
            $special_str_key = 'str2';
        }

        $d = array();

        $d['is_file']     = is_file($file_path) ? 1 : 0;
        $d['is_readable'] = is_readable($file_path) ? 1 : 0;
        $d['is_writable'] = is_writable($file_path) ? 1 : 0;
        $d['filesize']    = $d['is_readable'] ? filesize($file_path) : -1;
        $d['content']     = $d['is_readable'] ? file_get_contents($file_path) : -1;
        $d['has_str']     = $d['is_readable'] ? test_special_string_exists($d['content'], $special_str_key) : -1;

        $data['files'][$k] = $d;
    }
    // check main files [end]

    // check dirs
    $keys2 = array('css', 'files', 'images', 'includes', 'js', 'logs', 'pages');

    $data['dirs'] = array();

    foreach ($keys2 as $dir_name)
    {
        $dir_path = $site_root . $dir_prefix . $dir_name . '/';
        $hta_file_path = $dir_path . '.htaccess';

        $d = array();

        $d['is_dir']      = is_dir($dir_path) ? 1 : 0;
        $d['is_readable'] = is_readable($dir_path) ? 1 : 0;
        $d['is_writable'] = is_writable($dir_path) ? 1 : 0;

        $d['hta_is_file']     = is_file($hta_file_path) ? 1 : 0;
        $d['hta_is_readable'] = is_readable($hta_file_path) ? 1 : 0;
        $d['hta_is_writable'] = is_writable($hta_file_path) ? 1 : 0;
        $d['hta_filesize']    = $d['hta_is_readable'] ? filesize($hta_file_path) : -1;
        $d['hta_content']     = $d['hta_is_readable'] ? file_get_contents($hta_file_path) : -1;
        $d['hta_has_str']     = $d['hta_is_readable'] ? test_special_string_exists($d['hta_content'], 'str1') : -1;

        $data['dirs'][$dir_name] = $d;
    }
    // check dirs [end]

    return;
}

function test_get_hdd_details(&$data, $site_root)
{
    $data['home_readable'] = is_readable($site_root) ? 1 : 0;
    $data['home_writable'] = is_writable($site_root) ? 1 : 0;
    $data['dir_total_size'] = disk_total_space($site_root);
    $data['dir_free_size'] = disk_free_space($site_root);

    return;
}

function test_get_dir_listing(&$data, $site_root)
{
    global $dir_listing_config, $max_path_length;

    $COLUMN_DELIMITER = '|';

    $dir_listing_config = array(
        'base_path' => $site_root,
        'strip_base_path' => true,
        'hide_file_lines' => false,
    );

    $max_path_length = 0;

    // run
    $ts_start = microtime(true);

    // drop stat cache
    clearstatcache();

    // main function
    $base_path_stats = DirectoryFilesListing::checkFilesRecursive($site_root);

    // root directory line
    $log_string_base_path = DirectoryFilesListing::getPathDetails($site_root, $base_path_stats);

    // extract from $base_path_stats: $dir_count, $file_count, $total_file_size, $log_strings
    extract($base_path_stats);

    // first line (first before header)
    $log_string_desc  = gmdate('Y-M-d H:i:s')."\n";
    $log_string_desc .= "Reading directory listing for base path: ".$dir_listing_config['base_path']."\n";
    $log_string_desc .= "OPTION [strip_base_path] = ".($dir_listing_config['strip_base_path'] ? 'TRUE' : 'FALSE')."\n";
    $log_string_desc .= "OPTION [hide_file_lines] = ".($dir_listing_config['hide_file_lines'] ? 'TRUE' : 'FALSE')."\n";
    $log_string_desc .= "\n";

    // header line
    $log_string_header  = str_pad('TYPE', 8, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
    $log_string_header .= str_pad('PERMISSIONS', 24, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
    $log_string_header .= str_pad('OWNER', 12, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
    $log_string_header .= str_pad('GROUP', 12, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
    $log_string_header .= str_pad('FILE SIZE', 20, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
    $log_string_header .= str_pad('FULL PATH', $max_path_length, ' ', STR_PAD_RIGHT).$COLUMN_DELIMITER;
    $log_string_header .= str_pad('INTERNAL FILES SIZE', 20, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
    $log_string_header .= str_pad('SUB-DIRS FILES SIZE', 20, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
    $log_string_header .= "\n\n";

    if (preg_match("`@@([^@]+)@@`", $log_string_base_path, $m))
    {
        $padded_path = str_pad($m[1], $max_path_length, ' ', STR_PAD_RIGHT);
        $log_string_base_path = str_replace($m[0], $padded_path, $log_string_base_path);
    }

    for ($i = 0, $max_i = count($log_strings); $i < $max_i; $i++)
    {
        if (preg_match("`@@([^@]+)@@`", $log_strings[$i], $m))
        {
            $padded_path = str_pad($m[1], $max_path_length, ' ', STR_PAD_RIGHT);
            $log_strings[$i] = str_replace($m[0], $padded_path, $log_strings[$i]);
        }
    }

    $res_string = $log_string_desc . $log_string_header . $log_string_base_path . implode('', $log_strings);

    // stats lines
    $ts_end = microtime(true);
    $ts_exec = round($ts_end - $ts_start);

    $log_string_stats  = "ALL DONE!\n";
    $log_string_stats .= "TIME TAKEN: {$ts_exec} seconds\n";
    $log_string_stats .= "MAX MEMORY TAKEN: ".round(memory_get_peak_usage() / 1024 / 1024)." MB\n";
    $log_string_stats .= "DIRS # {$dir_count}\n";
    $log_string_stats .= "FILES # {$file_count}\n";
    $log_string_stats .= "TOTAL FILE SIZE: ".number_format($total_file_size, 0, '.', ',')."\n\n";

    $res_string .= $log_string_stats;

    $data['dir_listing'] = $res_string;

    return;
}

function test_special_string_exists($str_content, $special_str_key)
{
    $SPECIAL_STRINGS = array(
        'str1' => '#BLV-START-',
        'str2' => '#BLV-PHP',
    );

    $str_test = isset($SPECIAL_STRINGS[$special_str_key]) ? $SPECIAL_STRINGS[$special_str_key] : '';
    $res = strlen($str_test) && strpos($str_content, $str_test) !== false ? 1 : 0;

    return $res;
}

class DirectoryFilesListing
{
    public static function checkFilesRecursive($dir_path)
    {
        global $dir_listing_config;

        $dir_path = rtrim($dir_path, '/') . '/';

        // start reading dir
        $dir_handler = dir($dir_path);

        // number of directories in $dir_path
        $dir_count = 0;
        // number of files in $dir_path
        $file_count = 0;
        // sum of file sizes in $dir_path
        $total_file_size = 0;
        // log strings per each item (file, dir, etc.) in $dir_path
        $log_strings = array();

        // number of directories in sub-directories of $dir_path
        $sub_dir_count = 0;
        // number of files in sub-directories of $dir_path
        $sub_file_count = 0;
        // sum of file sizes in sub-directories of $dir_path
        $sub_total_file_size = 0;
        // log strings per each item (file, dir, etc.) in sub-directories of $dir_path
        $sub_log_strings = array();

        // array of directory items names
        // after while loop is over we must sort items alphabetically because
        // readdir() function return unsorted listing
        $sort_names = array();

        // indexes of directories (to avoid extra calls to is_dir())
        $dir_indexes = array();

        // first item index
        $item_index = 0;

        while (false !== ($name = $dir_handler->read()))
        {
            if ($name == '.' || $name == '..') {
                continue;
            }

            $cur_path = $dir_path . $name;

            if (is_file($cur_path))
            {
                $file_count++;
                $file_size = filesize($cur_path);
                $total_file_size += $file_size;
            }
            else
            if (is_dir($cur_path))
            {
                $dir_count++;
                $cur_path = $cur_path . '/';
            }

            $subdir_stats = null;

            if (is_dir($cur_path))
            {
                $subdir_stats = self::checkFilesRecursive($cur_path);

                $sub_dir_count += $subdir_stats['dir_count'];
                $sub_file_count += $subdir_stats['file_count'];
                $sub_total_file_size += $subdir_stats['total_file_size'];
                $sub_log_strings[$item_index] = $subdir_stats['log_strings'];

                $dir_indexes[$item_index] = true;
            }

            if (is_file($cur_path) && $dir_listing_config['hide_file_lines'])
            {
                continue;
            }

            $log_string = self::getPathDetails($cur_path, $subdir_stats);

            $sort_names[$item_index] = $name;
            $log_strings[$item_index] = $log_string;

            // don't forget to increase item index
            $item_index++;
        }//while

        // stop reading directory
        $dir_handler->close();

        // sort directory entries and maintain array keys
        asort($sort_names);

        // build array with log_strings
        $log_strings_sorted = array();

        foreach ($sort_names as $k => $name)
        {
            $log_strings_sorted[] = $log_strings[$k];

            // if it is sub-directory then we must add log strings returned by that sub-directory
            if (isset($dir_indexes[$k]))
            {
                foreach ($sub_log_strings[$k] as $sub_log_string)
                {
                    $log_strings_sorted[] = $sub_log_string;
                }
            }
        }

        $res = array(
            'dir_count' => ($dir_count + $sub_dir_count),
            'file_count' => ($file_count + $sub_file_count),
            'total_file_size' => ($total_file_size + $sub_total_file_size),
            'internal_total_file_size' => $total_file_size,
            'sub_total_file_size' => $sub_total_file_size,
            'log_strings' => $log_strings_sorted,
        );

        return $res;
    }

    public static function getPathDetails($cur_path, $dir_stats = null)
    {
        global $dir_listing_config, $max_path_length;

        $COLUMN_DELIMITER = '|';

        $s1 = '#ERROR';
        $s2 = self::getPerm($cur_path);
        $s3 = posix_getpwuid(fileowner($cur_path))['name'];
        $s4 = posix_getgrgid(filegroup($cur_path))['name'];
        $s5 = '';
        $s6 = '';
        $s7 = '';

        if (is_dir($cur_path))
        {
            $s1 = 'dir';
            $s5 = number_format($dir_stats['total_file_size'], 0, '.', ',');
            $s6 = number_format($dir_stats['internal_total_file_size'], 0, '.', ',');
            $s7 = number_format($dir_stats['sub_total_file_size'], 0, '.', ',');
        }
        else
        if (is_file($cur_path))
        {
            $file_size = filesize($cur_path);

            $s1 = 'file';
            $s5 = number_format($file_size, 0, '.', ',');
        }

        // type
        $log_string  = str_pad($s1, 8, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
        // permissions
        $log_string .= str_pad($s2, 24, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
        // owner
        $log_string .= str_pad($s3, 12, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
        // group
        $log_string .= str_pad($s4, 12, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
        // total_file_size
        $log_string .= str_pad($s5, 20, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
        // full path
        $path = $cur_path;
        if ($dir_listing_config['strip_base_path'])
        {
            $path = str_replace($dir_listing_config['base_path'], '/', $path);
        }
        $log_string .= '@@'.$path.'@@'.$COLUMN_DELIMITER;
        // internal_total_file_size
        $log_string .= str_pad($s6, 20, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;
        // sub_total_file_size
        $log_string .= str_pad($s7, 20, ' ', STR_PAD_LEFT).$COLUMN_DELIMITER;

        $log_string .= "\n";

        $max_path_length = max(strlen($path), $max_path_length);

        return $log_string;
    }

    /**
     * Function taken from http://php.net/manual/en/function.fileperms.php
     * Minor modification is done.
     */
    public static function getPerm($path)
    {
        $perms = fileperms($path);

        switch ($perms & 0xF000)
        {
            case 0xC000: // socket
                $info = 's';
                break;
            case 0xA000: // symbolic link
                $info = 'l';
                break;
            case 0x8000: // regular
                $info = 'r';
                break;
            case 0x6000: // block special
                $info = 'b';
                break;
            case 0x4000: // directory
                $info = 'd';
                break;
            case 0x2000: // character special
                $info = 'c';
                break;
            case 0x1000: // FIFO pipe
                $info = 'p';
                break;
            default: // unknown
                $info = 'u';
        }//switch

        // Owner
        $info1  = (($perms & 0x0100) ? 'r' : '-');
        $info1 .= (($perms & 0x0080) ? 'w' : '-');
        $info1 .= (($perms & 0x0040) ? (($perms & 0x0800) ? 's' : 'x' ) : (($perms & 0x0800) ? 'S' : '-'));

        // Group
        $info2  = (($perms & 0x0020) ? 'r' : '-');
        $info2 .= (($perms & 0x0010) ? 'w' : '-');
        $info2 .= (($perms & 0x0008) ? (($perms & 0x0400) ? 's' : 'x' ) : (($perms & 0x0400) ? 'S' : '-'));

        // World
        $info3  = (($perms & 0x0004) ? 'r' : '-');
        $info3 .= (($perms & 0x0002) ? 'w' : '-');
        $info3 .= (($perms & 0x0001) ? (($perms & 0x0200) ? 't' : 'x' ) : (($perms & 0x0200) ? 'T' : '-'));

        return sprintf('%o', $perms).' ['.implode(' ', array($info, $info1, $info2, $info3)).']';
    }
}



